﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins.Provider
{
    public class ProviderValidationReset : IPlugin
    {
        private const string PluginName = "ProviderReset";
        private const string ProcessDateAttribute = "ppms_validationcompletedate";
        private IOrganizationService _service;
        private ITracingService _tracingService;

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "account")
                    return;

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    var isReset = entity.GetAttributeValue<bool?>("ppms_validationreset");
                    if (!isReset.HasValue || !isReset.Value)
                    {
                        _tracingService.Trace("No reset detected. ({0})", entity.Id);
                        return;
                    }

                    // Get related entities to deactivate
                    _tracingService.Trace("Retrieve Provider and related entities");
                    var provider = GetProvider(_service, entity.Id);
                    if (provider == null)
                    {
                        _tracingService.Trace("Failed to retrieve provider related entities");
                        return;
                    }
                    _tracingService.Trace("Provider Retrieved: {0}", provider.Name);

                    IList<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    var isActive = PpmsHelper.IsActive(provider);
                    _tracingService.Trace("Check status: {0}", isActive);

                    if (!isActive)
                    {
                        // Get current batch details
                        IEnumerable<ppms_batchdetail> batchDetails = null;
                        if (provider.ppms_account_batchdetail_provider != null)
                            batchDetails = provider.ppms_account_batchdetail_provider.OrderBy(d => d.CreatedOn).Where(b => b.ppms_validationcompletedate == null);

                        if (batchDetails == null || !batchDetails.Any())
                        {
                            _tracingService.Trace("Batch detail does not exist.");
                            return;
                        }

                        _tracingService.Trace("Provider in Inactive Status");
                        // determine target batch detail
                        var statusReason = provider.GetAttributeValue<OptionSetValue>("statuscode");
                        if (statusReason != null)
                        {
                            switch (statusReason.Value)
                            {
                                case (int)Account_StatusCode.LEIEExclusion:
                                    _tracingService.Trace("Provider in Inactive Status");
                                    PpmsHelper.CreateBatchDetailLeieResult(batchDetails, _service);
                                    break;
                                case (int)Account_StatusCode.NPICheckFailure:
                                    _tracingService.Trace("Provider in Inactive Status");
                                    PpmsHelper.CreateBatchDetailNpiCheckResult(batchDetails, _service);
                                    break;
                                default:
                                    break;
                            }
                        }
                    }

                    // Set Batch Detail validation complete
                    requests = PpmsHelper.MarkBatchDetailAsComplete(provider, _service);

                    // Execute requests
                    if (requests.Count > 0)
                    {
                        _tracingService.Trace("Execute requests {0}", requests.Count);
                        foreach (var request in requests)
                        {
                            _service.Execute(request);
                        }
                    }
                    _tracingService.Trace("Execution complete.");
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        private Account GetProvider(IOrganizationService service, Guid providerId)
        {
            using (var context = new PpmsContext(service))
            {
                var account = context.AccountSet.FirstOrDefault(a => a.AccountId == providerId);
                context.LoadProperty(account, new Relationship("ppms_account_batchdetail_provider"));
                return account;
            }
        }

        private Entity GetBatchDetail(Guid batchDetailId)
        {
            return _service.Retrieve("ppms_batchdetail", batchDetailId, new ColumnSet(new string[] { "ppms_batchdetailid", "ppms_name", "ppms_batch", ProcessDateAttribute }));
        }

        private Account GetProviderWithBatchDetail(Guid providerId)
        {
            using (var context = new PpmsContext(_service))
            {
                var provider = context.AccountSet.FirstOrDefault(a => a.AccountId == providerId);
                if (provider != null)
                {
                    context.LoadProperty(provider, new Relationship("ppms_account_batchdetail_provider"));
                }
                return provider;
            }
        }

        private Entity GetCurrentBatchDetail(EntityReference providerRef)
        {
            if (providerRef != null)
            {
                var provider = GetProviderWithBatchDetail(providerRef.Id);
                if (provider != null)
                {
                    // determine target batch detail
                    var children = provider.ppms_account_batchdetail_provider;
                    if (children != null && children.Any())
                    {
                        return provider.ppms_account_batchdetail_provider.OrderBy(o => o.CreatedOn).Where(a => a.ppms_validationcompletedate == null).FirstOrDefault();
                    }
                }
            }

            return null;
        }

        private IEnumerable<Entity> GetCurrentBatchDetails(EntityReference providerRef)
        {
            if (providerRef != null)
            {
                var provider = GetProviderWithBatchDetail(providerRef.Id);
                if (provider != null)
                {
                    // determine target batch detail
                    var children = provider.ppms_account_batchdetail_provider;
                    if (children != null && children.Any())
                    {
                        return provider.ppms_account_batchdetail_provider.OrderBy(o => o.CreatedOn).Where(a => a.ppms_validationcompletedate == null);
                    }
                }
            }

            return null;
        }
    }
}
